/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/. */

// Test from official test vectors in
// https://github.com/usnistgov/ACVP-Server/tree/v1.1.0.35/gen-val/json-files/ML-KEM-keyGen-FIPS203
// Note that the encryption and decryption keys are replaced with their SHA3-256
// hash values.

#include <vector>

#include "kyber.h"

struct MlKem768KeyGen {
  KyberParams params;
  uint8_t seed[KYBER_KEYPAIR_COIN_BYTES];
  uint8_t publicKeyDigest[SHA3_256_LENGTH];
  uint8_t privateKeyDigest[SHA3_256_LENGTH];
};

std::vector<MlKem768KeyGen> MlKem768KeyGenTests = {
    {params_ml_kem768,
     {0xE3, 0x4A, 0x70, 0x1C, 0x4C, 0x87, 0x58, 0x2F, 0x42, 0x26, 0x4E,
      0xE4, 0x22, 0xD3, 0xC6, 0x84, 0xD9, 0x76, 0x11, 0xF2, 0x52, 0x3E,
      0xFE, 0x0C, 0x99, 0x8A, 0xF0, 0x50, 0x56, 0xD6, 0x93, 0xDC, 0xA8,
      0x57, 0x68, 0xF3, 0x48, 0x6B, 0xD3, 0x2A, 0x01, 0xBF, 0x9A, 0x8F,
      0x21, 0xEA, 0x93, 0x8E, 0x64, 0x8E, 0xAE, 0x4E, 0x54, 0x48, 0xC3,
      0x4C, 0x3E, 0xB8, 0x88, 0x20, 0xB1, 0x59, 0xEE, 0xDD},
     {0xE2, 0x90, 0x20, 0x83, 0x9D, 0x05, 0x2F, 0xA3, 0x72, 0x58, 0x56,
      0x27, 0xF8, 0xB5, 0x9E, 0xE3, 0x12, 0xAE, 0x41, 0x4C, 0x97, 0x9D,
      0x82, 0x5F, 0x06, 0xA6, 0x92, 0x9A, 0x79, 0x62, 0x57, 0x18},
     {0xA4, 0xE8, 0xBA, 0x80, 0xBB, 0x7A, 0x74, 0x5E, 0x93, 0x6D, 0x47,
      0x78, 0x4C, 0x07, 0xFF, 0xA6, 0xA3, 0x14, 0xCA, 0xF5, 0xA8, 0xDE,
      0xB4, 0x64, 0x8C, 0x5C, 0x2D, 0x6A, 0xE9, 0x30, 0xEB, 0xCD}},

    {params_ml_kem768,
     {0x44, 0x4F, 0x03, 0x2D, 0xD1, 0x9A, 0xE7, 0x51, 0x8C, 0x4B, 0x35,
      0xB0, 0x73, 0x2A, 0x41, 0xDC, 0x56, 0x78, 0x45, 0xAB, 0xA8, 0xBD,
      0x7B, 0x04, 0xA9, 0xC4, 0x13, 0xA0, 0xCF, 0x2D, 0xE0, 0xB5, 0xDF,
      0x0F, 0x28, 0x24, 0x11, 0xF4, 0xA0, 0x71, 0x48, 0x9A, 0x8F, 0x61,
      0x8E, 0x2A, 0xE5, 0xAE, 0xF4, 0x01, 0x31, 0xCA, 0xC5, 0x23, 0x3D,
      0x6D, 0x73, 0x15, 0x22, 0x72, 0x0C, 0x2F, 0xEB, 0x1C},
     {0xBB, 0xA2, 0x83, 0xF4, 0xC9, 0x93, 0xA0, 0x10, 0x08, 0x1E, 0x2C,
      0xC5, 0x71, 0xD9, 0x72, 0x34, 0x47, 0x2C, 0xC9, 0x85, 0x8D, 0x19,
      0x9C, 0xF0, 0xD6, 0xE6, 0xB9, 0xBD, 0x72, 0x0C, 0x26, 0x65},
     {0xF5, 0xFE, 0x55, 0x88, 0xA7, 0x0E, 0xD8, 0xBB, 0x7E, 0x74, 0x4A,
      0x3F, 0x46, 0xAA, 0x02, 0x00, 0xFA, 0x33, 0x3D, 0xBE, 0x21, 0x61,
      0x06, 0xD6, 0x27, 0x87, 0x14, 0x1D, 0xD5, 0xFC, 0x42, 0xD8}},

    {params_ml_kem768,
     {0x09, 0x22, 0x71, 0xD0, 0x5C, 0xA6, 0x3C, 0x60, 0x88, 0x0A, 0xF4,
      0x04, 0xD6, 0x0B, 0xC4, 0xBB, 0x95, 0x39, 0xE2, 0xEA, 0x12, 0x96,
      0x95, 0x81, 0x89, 0x8D, 0x56, 0xE0, 0xAC, 0x9A, 0x5A, 0x68, 0x5A,
      0xA6, 0xDC, 0x62, 0x0A, 0x6E, 0x9A, 0x60, 0xCF, 0x19, 0xA7, 0xB4,
      0xF0, 0xFF, 0x80, 0x5B, 0xDA, 0x82, 0x19, 0x52, 0x2A, 0x54, 0x8E,
      0xE5, 0x85, 0x7C, 0x3F, 0xF6, 0x06, 0x0C, 0x7A, 0x2F},
     {0x66, 0xE5, 0x24, 0x8C, 0xD3, 0x11, 0x28, 0x6D, 0x6D, 0xD0, 0x3E,
      0x01, 0x03, 0x91, 0xD9, 0x0D, 0x76, 0x04, 0x4B, 0xF4, 0x98, 0xB5,
      0x3C, 0x9D, 0x82, 0x02, 0xA9, 0xEB, 0x64, 0x35, 0x27, 0x39},
     {0x95, 0x79, 0x43, 0xE3, 0x10, 0x99, 0xB0, 0x23, 0xAF, 0x68, 0xD0,
      0xFA, 0xB9, 0x08, 0x9C, 0x08, 0xCD, 0xDF, 0x34, 0x6B, 0xE9, 0xBC,
      0x6D, 0xAD, 0xDD, 0x72, 0x61, 0x48, 0xEB, 0x7C, 0x8D, 0xC8}},

    {params_ml_kem768,
     {0xBB, 0xF7, 0x57, 0x4C, 0xF5, 0xF3, 0x2B, 0xE4, 0x9E, 0x1F, 0x39,
      0xCE, 0x33, 0x87, 0x0D, 0x9D, 0x63, 0x84, 0x05, 0x6D, 0x60, 0xD2,
      0x23, 0x00, 0x3B, 0x6B, 0x0C, 0x10, 0xD5, 0xC4, 0x21, 0x80, 0x7C,
      0xF5, 0x0F, 0x72, 0x37, 0xA9, 0x70, 0x72, 0xF0, 0x3F, 0x31, 0xCF,
      0xD5, 0x9F, 0xA8, 0xE8, 0x63, 0xBC, 0xA3, 0xAF, 0x73, 0x75, 0xE0,
      0xCA, 0x69, 0x8F, 0xF6, 0x65, 0x66, 0x1C, 0x24, 0xCF},
     {0xA9, 0x18, 0xB3, 0x9F, 0x71, 0xBB, 0xB2, 0xC1, 0x0D, 0xB3, 0x56,
      0x39, 0xE5, 0xFD, 0x2C, 0xE6, 0x21, 0x86, 0x8C, 0xC0, 0x21, 0x49,
      0xE0, 0x29, 0xEB, 0x47, 0x89, 0x94, 0x07, 0xD9, 0x63, 0x00},
     {0x7A, 0x6E, 0x99, 0xA1, 0x7F, 0x7B, 0x9A, 0x7A, 0x7C, 0xFF, 0x34,
      0xB7, 0x3F, 0x5C, 0xD2, 0x15, 0x62, 0xA1, 0x73, 0x66, 0x73, 0x22,
      0x6B, 0xF0, 0x7E, 0xB4, 0x6E, 0xDB, 0xCC, 0xC6, 0x9A, 0x3E}},

    {params_ml_kem768,
     {0xD1, 0x2C, 0xD9, 0xB6, 0x5B, 0x7C, 0x58, 0xB2, 0x19, 0x5A, 0xE0,
      0xBE, 0x02, 0x82, 0x52, 0x7B, 0xAC, 0x06, 0xC2, 0xD2, 0x5C, 0xB0,
      0x47, 0x26, 0x28, 0xD6, 0x47, 0x15, 0xF7, 0xF6, 0xA3, 0x78, 0xC5,
      0x93, 0x62, 0x78, 0x07, 0x07, 0x46, 0x84, 0xB7, 0xD3, 0x63, 0x44,
      0x1F, 0x80, 0xF6, 0xA3, 0xD1, 0x85, 0xD6, 0x78, 0x78, 0x70, 0x2D,
      0x33, 0xA4, 0xE0, 0xBD, 0xA2, 0x00, 0x0F, 0x85, 0x7D},
     {0xC8, 0x6A, 0x41, 0xEF, 0xD3, 0x15, 0x19, 0x1F, 0x24, 0xD2, 0xE6,
      0xBD, 0xD8, 0x74, 0x33, 0xD5, 0x13, 0x3D, 0x67, 0x34, 0xFB, 0xEA,
      0xA9, 0xDA, 0x80, 0x43, 0xD9, 0x19, 0x50, 0x00, 0x00, 0x48},
     {0xFA, 0x4C, 0x8A, 0x8E, 0xE6, 0xE6, 0x89, 0xD2, 0x88, 0x1B, 0x18,
      0x8F, 0x0B, 0xFA, 0xB4, 0xFC, 0x94, 0x16, 0x2C, 0x9E, 0x74, 0xD3,
      0x7A, 0x67, 0x17, 0x4A, 0x4A, 0xA0, 0xC9, 0xA6, 0x4D, 0xB7}},

    {params_ml_kem768,
     {0x79, 0xC0, 0x06, 0xD5, 0x47, 0x0C, 0x22, 0x9A, 0xFC, 0xE7, 0x58,
      0x85, 0x46, 0xE5, 0x22, 0x04, 0xB0, 0x9F, 0x50, 0x86, 0x97, 0x48,
      0x65, 0xB4, 0x26, 0xAA, 0xAA, 0x19, 0x8C, 0x6C, 0xBA, 0x7A, 0xE0,
      0x17, 0x02, 0xE1, 0x22, 0x8F, 0x53, 0x0A, 0xC9, 0x6D, 0xB0, 0x53,
      0xA4, 0x15, 0xBE, 0x97, 0x74, 0x9A, 0x10, 0x9A, 0x1F, 0xD4, 0x05,
      0x7B, 0xA1, 0x28, 0x64, 0x9B, 0x17, 0xEC, 0x07, 0xAD},
     {0x32, 0xF4, 0x34, 0x78, 0x3F, 0x38, 0xED, 0x27, 0x73, 0x82, 0xAA,
      0x17, 0xAC, 0xF5, 0xFE, 0xC8, 0x7E, 0x72, 0xBE, 0xF7, 0x29, 0xA6,
      0x3E, 0x69, 0xAF, 0x73, 0x87, 0xE9, 0xCC, 0x5B, 0xB3, 0x39},
     {0x70, 0x03, 0x90, 0xB9, 0x7F, 0xBA, 0xDC, 0x59, 0xDC, 0xB9, 0x06,
      0x32, 0x86, 0x82, 0xCA, 0x37, 0x77, 0xEC, 0x1F, 0xDE, 0x8B, 0xB3,
      0x23, 0x59, 0xEF, 0xC8, 0xF7, 0xD6, 0xE6, 0x8A, 0xB5, 0xFE}},

    {params_ml_kem768,
     {0xB0, 0x4F, 0x63, 0x1B, 0x33, 0x0D, 0x83, 0x99, 0x1B, 0x5C, 0x01,
      0xE7, 0xF6, 0x94, 0x52, 0xDF, 0xC3, 0x94, 0xF9, 0x68, 0x96, 0x32,
      0xF8, 0xC7, 0xF6, 0x0D, 0xBF, 0xAB, 0x92, 0xA9, 0xCE, 0xA5, 0xAE,
      0x51, 0x63, 0x9E, 0xF7, 0xF2, 0x6F, 0xD2, 0x21, 0x5A, 0xD1, 0x1C,
      0xBE, 0x1E, 0xDE, 0xB3, 0xB9, 0x43, 0xD6, 0x68, 0xEE, 0xEF, 0xEE,
      0x13, 0xED, 0x5B, 0x0D, 0xA3, 0xE0, 0xA5, 0xF3, 0xED},
     {0x44, 0x13, 0x2D, 0x7C, 0xEA, 0x4F, 0x7C, 0xB9, 0xB0, 0x6A, 0xA5,
      0x9C, 0x42, 0x13, 0xFA, 0x62, 0x93, 0x56, 0x3C, 0x45, 0x16, 0xCF,
      0x03, 0x34, 0x91, 0x74, 0x2C, 0x38, 0x9A, 0xF3, 0x86, 0x43},
     {0x13, 0xD9, 0x42, 0xF7, 0xED, 0xF8, 0x9A, 0xDC, 0x56, 0x5C, 0xDB,
      0x2B, 0x90, 0xE1, 0x0B, 0xAD, 0xEC, 0x8C, 0xAA, 0x49, 0x62, 0x17,
      0x68, 0x70, 0x3A, 0xF5, 0x45, 0x5A, 0x77, 0xE7, 0x81, 0x72}},

    {params_ml_kem768,
     {0x3D, 0x63, 0xBD, 0x6C, 0x31, 0x0A, 0xFC, 0xF6, 0x84, 0x29, 0x2E,
      0x5F, 0x8E, 0x1B, 0x98, 0xCC, 0x75, 0xB5, 0xA2, 0x7B, 0x21, 0x52,
      0x62, 0x68, 0x44, 0x41, 0x44, 0xAB, 0x24, 0xAB, 0x29, 0x67, 0x6F,
      0x9F, 0xF5, 0x65, 0x4F, 0xDA, 0x78, 0x77, 0x44, 0x98, 0xE2, 0x64,
      0x3E, 0x93, 0x5D, 0x21, 0x41, 0x2C, 0xEB, 0x49, 0xBC, 0x39, 0x35,
      0x32, 0xC8, 0x0C, 0x47, 0xA9, 0x82, 0x41, 0x8F, 0x66},
     {0xD4, 0xF2, 0xA9, 0xB4, 0x85, 0xFF, 0xC5, 0x44, 0xCD, 0x3D, 0xF6,
      0x7D, 0x23, 0xC8, 0x01, 0x50, 0xAA, 0xF7, 0xA4, 0x5C, 0xD9, 0x46,
      0xF4, 0xB7, 0xDB, 0x2B, 0x67, 0xF4, 0xF8, 0xB2, 0x22, 0x53},
     {0x65, 0x25, 0x5B, 0x78, 0x9A, 0x21, 0x1E, 0xE9, 0xB8, 0x13, 0x94,
      0xD8, 0x5F, 0x4C, 0xE5, 0xB0, 0xF6, 0x74, 0x01, 0xAE, 0xED, 0x8D,
      0x77, 0x9E, 0x8F, 0x41, 0xA1, 0x6A, 0x51, 0x03, 0x95, 0x11}},

    {params_ml_kem768,
     {0x24, 0x9D, 0x48, 0x94, 0x1A, 0xBC, 0x01, 0xC9, 0x29, 0x07, 0x19,
      0xFB, 0x34, 0xD9, 0x1B, 0x05, 0xE7, 0x74, 0xE7, 0x0E, 0x6F, 0x01,
      0x81, 0xE1, 0x78, 0x3F, 0x25, 0x86, 0xE2, 0x49, 0x95, 0x36, 0xD0,
      0x83, 0xE6, 0x92, 0x2E, 0xF0, 0xA8, 0x18, 0x30, 0x8F, 0xD7, 0xFE,
      0x7C, 0xF5, 0xAD, 0x3A, 0x96, 0x94, 0x24, 0x42, 0xBE, 0x32, 0x7B,
      0x0A, 0x30, 0x76, 0x85, 0xC2, 0xD4, 0x31, 0x59, 0x01},
     {0x5D, 0x0B, 0xB5, 0xF5, 0x14, 0xCA, 0xC1, 0x67, 0xBB, 0x2E, 0x2B,
      0x5F, 0xE9, 0x89, 0xCE, 0x88, 0xED, 0x65, 0x31, 0x5B, 0xC6, 0x10,
      0xD9, 0xA5, 0xBC, 0xC7, 0x7B, 0xA8, 0x0D, 0xFA, 0x2F, 0xF1},
     {0x89, 0x38, 0x06, 0x3D, 0xCD, 0xFE, 0x53, 0x64, 0xEE, 0x8B, 0xC9,
      0x76, 0xE5, 0x1B, 0x17, 0xE9, 0xE4, 0x60, 0xC5, 0xD1, 0x69, 0x55,
      0x14, 0x6A, 0x3F, 0xC1, 0xF1, 0xCF, 0xFF, 0x40, 0x09, 0x79}},

    {params_ml_kem768,
     {0xE1, 0xCF, 0xB8, 0x19, 0x58, 0x77, 0xB2, 0xD4, 0xFF, 0x33, 0x63,
      0xBA, 0xC3, 0xB4, 0xE7, 0xBE, 0xBA, 0x6D, 0xC3, 0xCB, 0xB7, 0x89,
      0xB1, 0xB2, 0x42, 0x15, 0x39, 0x3F, 0x6C, 0x9B, 0xBF, 0xAE, 0xA2,
      0x0A, 0xBA, 0x8A, 0x8D, 0xDC, 0x21, 0x2D, 0xE8, 0x25, 0xBE, 0x0D,
      0x3B, 0xE5, 0x77, 0x01, 0xA6, 0xB5, 0xB3, 0xA4, 0x6A, 0x30, 0x0D,
      0x9B, 0x59, 0x45, 0xF5, 0x79, 0xA5, 0x9A, 0xFA, 0xBE},
     {0xB5, 0xE9, 0x64, 0x69, 0x5C, 0x24, 0xF5, 0x7C, 0xD0, 0x5B, 0x8B,
      0xDC, 0x23, 0x94, 0x9D, 0x38, 0x2C, 0x7E, 0x90, 0x23, 0xCC, 0x14,
      0x32, 0xBC, 0x13, 0x16, 0x89, 0x52, 0x8B, 0x14, 0x53, 0xB0},
     {0xED, 0xAB, 0xC0, 0x2A, 0xAD, 0x2E, 0x29, 0x9F, 0xCD, 0x21, 0xA0,
      0x97, 0x31, 0xCC, 0xAD, 0x4A, 0xE2, 0x13, 0xC0, 0xEF, 0xD1, 0x23,
      0xE7, 0x9E, 0x06, 0xF5, 0xFA, 0x35, 0x83, 0x90, 0xFD, 0xC6}},

    {params_ml_kem768,
     {0xAD, 0xC4, 0xDA, 0x59, 0xD9, 0x35, 0xDD, 0x87, 0x42, 0x0A, 0xCE,
      0xE5, 0x2A, 0xEE, 0x19, 0xCB, 0x37, 0x1F, 0xD0, 0xBB, 0x49, 0x8D,
      0x79, 0xBA, 0x68, 0x01, 0x59, 0xEF, 0x7C, 0xE3, 0x7C, 0x17, 0x7F,
      0xB9, 0x50, 0xA8, 0xF5, 0x1D, 0xCE, 0xC4, 0xBC, 0x7A, 0x57, 0x3E,
      0xDD, 0xA5, 0x6E, 0xCC, 0x04, 0x9E, 0x56, 0x88, 0x47, 0x6B, 0xD5,
      0xFD, 0x6C, 0xD0, 0x76, 0xA8, 0xF9, 0x9A, 0x01, 0x9A},
     {0xBA, 0xF1, 0x8B, 0x5A, 0x25, 0x08, 0x1C, 0x8A, 0x9F, 0x52, 0x61,
      0x11, 0xB6, 0x00, 0x95, 0x4D, 0x39, 0xBA, 0xDA, 0xB9, 0x04, 0x4F,
      0x59, 0x90, 0x3D, 0x2A, 0x8F, 0x21, 0xF8, 0xE1, 0xD7, 0x8B},
     {0x1F, 0x97, 0xFF, 0x7E, 0x34, 0x6C, 0x22, 0xA5, 0x1C, 0xC8, 0xAB,
      0x22, 0xD4, 0x7C, 0x23, 0x02, 0xE4, 0x0F, 0xB6, 0x73, 0xF2, 0xD3,
      0xAB, 0x74, 0x4B, 0x28, 0x77, 0x73, 0x28, 0x06, 0x2E, 0xB8}},

    {params_ml_kem768,
     {0x76, 0xCD, 0xCA, 0x53, 0xF7, 0x81, 0x80, 0x6D, 0x55, 0xCA, 0x8D,
      0x3B, 0xAF, 0xB3, 0xF4, 0xD3, 0x89, 0xD7, 0x12, 0xF1, 0x22, 0x1E,
      0x85, 0xB5, 0xE2, 0x9D, 0x6A, 0x46, 0x58, 0x0F, 0x97, 0x8C, 0x51,
      0xD5, 0x09, 0xCF, 0x26, 0x79, 0x97, 0x41, 0x63, 0x10, 0x99, 0x03,
      0x9F, 0x71, 0x3B, 0x22, 0x55, 0x1E, 0x2B, 0x0F, 0x02, 0x97, 0xBB,
      0x80, 0x9D, 0xF0, 0xCC, 0x8F, 0xC3, 0xE4, 0x7E, 0xEE},
     {0xCE, 0xFB, 0x59, 0x3C, 0x11, 0xED, 0x36, 0x0F, 0x40, 0x47, 0x32,
      0xEA, 0x8B, 0x65, 0x42, 0xFA, 0x97, 0x96, 0xF2, 0xAE, 0xBB, 0x4C,
      0x61, 0xEE, 0xA4, 0x0B, 0x6D, 0x8A, 0x59, 0x9C, 0x7F, 0x13},
     {0xEA, 0x7D, 0xA3, 0xF2, 0x6F, 0xA2, 0xFF, 0x6D, 0xFA, 0x09, 0x48,
      0xE8, 0x35, 0xA5, 0x46, 0xB4, 0x0C, 0x51, 0x93, 0x87, 0xAE, 0xD4,
      0xC5, 0x35, 0xD6, 0x04, 0x27, 0x67, 0x21, 0x8F, 0x06, 0x98}},

    {params_ml_kem768,
     {0x78, 0xAB, 0x6C, 0x49, 0x35, 0x4A, 0x01, 0x8B, 0xD3, 0x8A, 0x39,
      0x92, 0x6F, 0x82, 0x2A, 0x1A, 0xC4, 0xAC, 0xC4, 0xFF, 0x32, 0xDF,
      0xD7, 0xC0, 0x47, 0xCE, 0x08, 0x87, 0xA3, 0xAC, 0x18, 0x2C, 0x9C,
      0x33, 0x0A, 0xB4, 0x25, 0x7D, 0x7B, 0x87, 0xC4, 0x74, 0x2C, 0x6E,
      0x95, 0xB6, 0x6B, 0xDF, 0x80, 0x5C, 0x6A, 0x14, 0x5B, 0xF4, 0x44,
      0x83, 0x60, 0x92, 0xC6, 0xB1, 0xD2, 0xC5, 0xFF, 0xFF},
     {0xA8, 0x60, 0x4C, 0xD9, 0x0A, 0xAF, 0x5F, 0xB9, 0xBD, 0xA2, 0x20,
      0x81, 0x40, 0x69, 0xAA, 0x00, 0xCB, 0x5B, 0x5F, 0xFB, 0x7B, 0x60,
      0xE4, 0xBC, 0xC8, 0x6F, 0x16, 0xED, 0x0B, 0x49, 0xBA, 0x9B},
     {0x1A, 0xA9, 0xD0, 0x64, 0x03, 0x2E, 0x42, 0x2A, 0x80, 0xD9, 0x41,
      0x63, 0x1C, 0x06, 0xFD, 0x93, 0x86, 0x94, 0x5B, 0x63, 0x19, 0x7C,
      0xB7, 0xFF, 0x13, 0xEF, 0xDC, 0x28, 0x0E, 0xEF, 0xDE, 0xF2}},

    {params_ml_kem768,
     {0x13, 0xB7, 0x56, 0x20, 0xE4, 0xCB, 0x9A, 0xB9, 0xA6, 0x68, 0x9F,
      0x6E, 0x2B, 0xE4, 0x46, 0x39, 0xBA, 0xE6, 0xC9, 0xCB, 0x7D, 0xD6,
      0x41, 0xAC, 0x1C, 0x93, 0x77, 0x24, 0x2D, 0x99, 0x67, 0x9A, 0x18,
      0xEA, 0x1C, 0x75, 0x32, 0xF7, 0x06, 0xB0, 0x68, 0x70, 0xD0, 0xA1,
      0x04, 0x7A, 0xAE, 0x33, 0xD9, 0xE1, 0xFF, 0x9E, 0x9B, 0xCB, 0xBD,
      0x30, 0x2D, 0x88, 0x17, 0xEB, 0x7B, 0x02, 0x2A, 0x77},
     {0x17, 0x83, 0x91, 0x31, 0x32, 0xF0, 0x97, 0x61, 0x8B, 0xB3, 0x9B,
      0xD4, 0x74, 0x8B, 0x4E, 0xFE, 0x63, 0xDA, 0x07, 0xC2, 0x66, 0x97,
      0xF9, 0xB2, 0xF4, 0xE0, 0x6C, 0xB2, 0xD2, 0x70, 0x12, 0xAE},
     {0x6D, 0xBF, 0x71, 0x47, 0x32, 0x16, 0x5B, 0x21, 0xAC, 0x6C, 0x95,
      0xCF, 0xF0, 0xE4, 0x9F, 0x3B, 0x6B, 0x51, 0x60, 0x64, 0xC6, 0x17,
      0x4E, 0x14, 0x6E, 0xCF, 0x3F, 0x92, 0x40, 0x54, 0x49, 0x83}},

    {params_ml_kem768,
     {0x7C, 0x34, 0x58, 0x19, 0xC7, 0xC3, 0x27, 0xAD, 0x95, 0x71, 0xE5,
      0xDF, 0x88, 0x24, 0x49, 0xDB, 0x24, 0x38, 0x70, 0xD6, 0x86, 0xA9,
      0x76, 0x4D, 0x41, 0x29, 0xB2, 0x1E, 0x17, 0xAC, 0x86, 0xA9, 0xC7,
      0x1F, 0x7E, 0x44, 0x29, 0x59, 0x78, 0xFC, 0x63, 0xBF, 0x8F, 0x6A,
      0x68, 0xF8, 0x60, 0x9E, 0x98, 0xD1, 0x55, 0xFD, 0x7A, 0x74, 0xE1,
      0xFB, 0x79, 0x82, 0x73, 0x3F, 0xBF, 0x8A, 0x6C, 0x25},
     {0x3B, 0x1D, 0x86, 0x1C, 0x34, 0xDA, 0x18, 0x2B, 0xF4, 0xDD, 0x68,
      0x3A, 0xBE, 0x8D, 0x24, 0x78, 0x98, 0xE7, 0x1E, 0x95, 0xE2, 0x7A,
      0xF7, 0x24, 0x94, 0xC0, 0x2B, 0xA6, 0xFF, 0x3C, 0x81, 0x47},
     {0x44, 0xEE, 0xE4, 0x8A, 0x98, 0x1C, 0x89, 0x74, 0x0D, 0x8D, 0xD2,
      0x33, 0xFF, 0x49, 0x30, 0xD0, 0x3A, 0xD0, 0xBB, 0xBC, 0x63, 0x71,
      0x76, 0x13, 0x5C, 0x12, 0xAC, 0x9F, 0x77, 0x85, 0xF1, 0x4B}},

    {params_ml_kem768,
     {0x8D, 0x6D, 0xF2, 0xEB, 0x3D, 0xDA, 0xF9, 0x61, 0xFE, 0x5E, 0xB5,
      0x56, 0x84, 0x2B, 0x75, 0x8B, 0xEB, 0xC7, 0xEC, 0xB3, 0x12, 0xB6,
      0xD4, 0x62, 0x8B, 0x32, 0x3F, 0x48, 0x3B, 0x77, 0xD6, 0xF9, 0xEF,
      0x66, 0x8F, 0xB4, 0x1F, 0x49, 0xE8, 0x2E, 0xE0, 0xFE, 0x00, 0x91,
      0x9C, 0xC0, 0x65, 0x07, 0x54, 0x83, 0x21, 0x59, 0x3A, 0x7E, 0xCD,
      0x1D, 0x21, 0x12, 0x34, 0x26, 0x08, 0xD9, 0x5F, 0xFF},
     {0x84, 0x7F, 0x52, 0xD9, 0x58, 0x7D, 0xA7, 0xDD, 0x37, 0xF7, 0xAE,
      0x07, 0xBF, 0x1B, 0x9D, 0x4C, 0x94, 0xF0, 0x3C, 0x70, 0x23, 0x51,
      0xFB, 0x4C, 0x5A, 0xF4, 0x20, 0x0E, 0xFC, 0xA0, 0x7F, 0x38},
     {0x89, 0x8C, 0x9B, 0x7F, 0xBC, 0xB8, 0xC1, 0x1E, 0xF3, 0xBF, 0xEA,
      0x1B, 0x18, 0x81, 0x37, 0x19, 0x07, 0x63, 0x92, 0xF9, 0x5D, 0x43,
      0x05, 0x24, 0x4A, 0x14, 0x37, 0x30, 0xC8, 0x8B, 0xF1, 0x96}},

    {params_ml_kem768,
     {0xDB, 0x4E, 0xD8, 0xE9, 0xC3, 0xE1, 0xAC, 0x7A, 0x35, 0xEA, 0x4B,
      0x67, 0xA4, 0xEF, 0xCF, 0xB4, 0x69, 0x72, 0xA9, 0x84, 0xD1, 0x61,
      0xF7, 0x9F, 0x08, 0x41, 0x25, 0xD6, 0xD4, 0xAE, 0xE7, 0xAF, 0x26,
      0x34, 0x59, 0x37, 0xAD, 0xC9, 0x10, 0x41, 0x55, 0x27, 0x5E, 0x71,
      0x14, 0xE9, 0x3D, 0x9F, 0x58, 0x47, 0xEE, 0xA7, 0x3A, 0x93, 0x59,
      0x35, 0x85, 0x85, 0xB2, 0xD4, 0x23, 0x01, 0xA2, 0x94},
     {0x16, 0x16, 0x11, 0x13, 0xDF, 0x64, 0x68, 0x37, 0xA2, 0x88, 0x18,
      0xD9, 0xC3, 0x4E, 0xDA, 0xD5, 0x74, 0x72, 0x94, 0x45, 0x28, 0xFF,
      0xBE, 0xC6, 0xB1, 0xBD, 0x20, 0x42, 0x62, 0xDC, 0xA0, 0x4F},
     {0x0B, 0xFD, 0xE4, 0xEF, 0xFD, 0x32, 0x7C, 0x5F, 0x96, 0x98, 0x2F,
      0xA2, 0xAC, 0x30, 0x53, 0x02, 0xF2, 0x8E, 0x2A, 0xE2, 0x7B, 0x26,
      0x62, 0x8A, 0x39, 0x76, 0xA4, 0x13, 0x71, 0x22, 0x21, 0x1F}},

    {params_ml_kem768,
     {0xC6, 0xEF, 0xA7, 0xD5, 0xD5, 0x00, 0xE5, 0xBF, 0x85, 0x7D, 0x80,
      0xEA, 0xE2, 0xA6, 0xEE, 0x64, 0x14, 0x15, 0x99, 0x47, 0xFD, 0x4B,
      0xE5, 0x89, 0x35, 0x07, 0x24, 0xFA, 0xE5, 0xE5, 0x18, 0x05, 0x63,
      0x43, 0x5E, 0x06, 0xC2, 0xAA, 0x3D, 0xFB, 0x34, 0x77, 0x12, 0x07,
      0x10, 0xD5, 0xE7, 0xFF, 0x0D, 0xC0, 0xDA, 0x68, 0xD4, 0x64, 0x4A,
      0x24, 0xF6, 0x6A, 0x80, 0x12, 0xFB, 0x19, 0x36, 0x97},
     {0x0B, 0x2C, 0xEE, 0x55, 0xAB, 0x09, 0xD3, 0x3B, 0xEB, 0xC1, 0x11,
      0x9E, 0x3D, 0x82, 0x68, 0xD3, 0x21, 0xCE, 0x67, 0x5C, 0xA8, 0x23,
      0x3E, 0x6A, 0xEE, 0x59, 0x8C, 0x76, 0x52, 0x29, 0x8B, 0x01},
     {0xD0, 0x21, 0xA2, 0xC5, 0x5E, 0xB4, 0x48, 0xDC, 0x02, 0x24, 0xEA,
      0xDE, 0x09, 0xAC, 0xA2, 0xAE, 0xB2, 0x10, 0xEA, 0xCB, 0x65, 0x28,
      0x45, 0x5B, 0xC0, 0x26, 0xE6, 0xE4, 0x10, 0x26, 0xE2, 0x87}},

    {params_ml_kem768,
     {0x20, 0x85, 0x9B, 0x01, 0xDF, 0xC6, 0x0B, 0x61, 0x09, 0xE0, 0x23,
      0x4F, 0x3C, 0xAC, 0x7A, 0x24, 0x7D, 0x83, 0x86, 0x09, 0x9D, 0x83,
      0xD2, 0xD4, 0x47, 0xE9, 0xA2, 0x1A, 0xF9, 0xDE, 0x48, 0xBD, 0x8C,
      0x29, 0x42, 0xB7, 0x20, 0x7C, 0x2C, 0x59, 0xBD, 0x56, 0xFF, 0x9E,
      0xE0, 0xB1, 0x20, 0xB1, 0xDA, 0xD8, 0x1B, 0x05, 0x60, 0x26, 0x23,
      0x62, 0x3C, 0xBC, 0x7E, 0x0C, 0x20, 0xC9, 0xB7, 0x09},
     {0xEA, 0xFE, 0x2B, 0x26, 0xCB, 0x96, 0xB9, 0x7C, 0x22, 0x56, 0x4B,
      0x28, 0x32, 0x9B, 0x64, 0xA2, 0x06, 0x33, 0x1F, 0xF8, 0x42, 0xBF,
      0xED, 0x4A, 0xDF, 0xE3, 0xC7, 0xA0, 0xC4, 0xA4, 0x71, 0xBA},
     {0x28, 0xD7, 0xD2, 0x11, 0xAA, 0x16, 0xCF, 0x0D, 0x64, 0xAE, 0x44,
      0x14, 0xA3, 0x0F, 0x7D, 0x90, 0x00, 0x49, 0xC2, 0xE8, 0x70, 0x1B,
      0x26, 0x83, 0x3D, 0x18, 0x08, 0x7C, 0xED, 0x03, 0x49, 0xD6}},

    {params_ml_kem768,
     {0x40, 0x9E, 0x9F, 0x3A, 0xB5, 0x8D, 0x73, 0x6E, 0x12, 0x2E, 0xFC,
      0xC4, 0x24, 0x0B, 0xF8, 0x38, 0x8F, 0xDF, 0xDA, 0x67, 0x59, 0x00,
      0x4D, 0x42, 0x45, 0x70, 0x18, 0x01, 0x4A, 0x33, 0x5B, 0xE4, 0xEA,
      0xE3, 0x18, 0x34, 0x1D, 0x06, 0xE0, 0x80, 0x1C, 0x0C, 0xA4, 0xB8,
      0x73, 0x52, 0x0C, 0x71, 0x47, 0x40, 0xAD, 0x01, 0x7F, 0xE5, 0xA1,
      0x58, 0xD3, 0xBD, 0x40, 0x96, 0x0D, 0x90, 0x7A, 0xB7},
     {0x9E, 0x2F, 0xE7, 0xDD, 0x64, 0x6C, 0x14, 0x54, 0x84, 0xE1, 0x63,
      0xD6, 0xC3, 0x6D, 0xC6, 0xEA, 0x5D, 0x80, 0x2A, 0x0E, 0xEE, 0x6A,
      0xDA, 0xC9, 0x32, 0xC2, 0x0F, 0xDA, 0xAB, 0xB8, 0xBD, 0xD1},
     {0x90, 0x0B, 0xA6, 0xC8, 0xEE, 0x62, 0x0F, 0x64, 0x44, 0xE7, 0x66,
      0xDD, 0x1A, 0x30, 0xD2, 0xDD, 0xE6, 0x52, 0xEE, 0x68, 0x7E, 0x83,
      0x1A, 0xB7, 0x00, 0x48, 0xE8, 0x77, 0x0A, 0x77, 0x4D, 0x45}},

    {params_ml_kem768,
     {0xCE, 0x2C, 0xAC, 0xEB, 0xD5, 0x4A, 0xF1, 0xB4, 0xE7, 0x15, 0x88,
      0xDE, 0x9F, 0x22, 0xA6, 0xAF, 0x2C, 0x2E, 0x2A, 0xD7, 0xFD, 0x66,
      0xB9, 0xFE, 0xC0, 0xDF, 0x19, 0x18, 0x2E, 0x7F, 0x57, 0xEC, 0xEF,
      0x38, 0x26, 0x45, 0x20, 0x68, 0x50, 0x80, 0xF5, 0x29, 0x75, 0xBC,
      0x95, 0x7C, 0x5F, 0xB6, 0x09, 0xFB, 0x0E, 0x1B, 0xD0, 0x6D, 0x26,
      0xF5, 0x72, 0xCC, 0x54, 0x25, 0xCA, 0xE7, 0xDE, 0x5C},
     {0xA5, 0xA6, 0x67, 0x16, 0xD0, 0x11, 0xEE, 0xDF, 0x9E, 0x6A, 0x54,
      0x1F, 0x94, 0x38, 0xF8, 0x30, 0x96, 0x60, 0x65, 0x7E, 0xAF, 0xFC,
      0xDB, 0x01, 0xA1, 0x72, 0x99, 0x8E, 0x56, 0xD9, 0xA6, 0x0B},
     {0x57, 0x7D, 0xBF, 0xD4, 0x7D, 0xEC, 0xEC, 0x40, 0xAA, 0xFF, 0xB1,
      0xF2, 0x0F, 0x20, 0x2D, 0x9F, 0x52, 0xD5, 0xA9, 0x06, 0x2A, 0x21,
      0x85, 0x26, 0x40, 0x28, 0xD5, 0xAE, 0xD8, 0x2F, 0x50, 0x61}},

    {params_ml_kem768,
     {0x7E, 0x03, 0x01, 0x5C, 0x5D, 0x55, 0xFD, 0x98, 0x88, 0xE7, 0x30,
      0xC1, 0xE6, 0x0F, 0x90, 0xC5, 0xF6, 0xC2, 0xE3, 0xB1, 0xE8, 0xC7,
      0xC0, 0x8D, 0x86, 0x9F, 0x0C, 0x1D, 0x15, 0xB5, 0x40, 0xED, 0x17,
      0xE5, 0xAE, 0x70, 0x77, 0x16, 0x74, 0xBE, 0x89, 0x03, 0xCC, 0x21,
      0xB3, 0xA9, 0x02, 0x48, 0xD9, 0x93, 0xC2, 0x61, 0xB6, 0xCE, 0xEF,
      0x2C, 0x74, 0x78, 0x73, 0xD1, 0x13, 0x86, 0x9B, 0x55},
     {0x6A, 0x22, 0xA9, 0xBE, 0x6B, 0x0A, 0x57, 0xE5, 0x9B, 0x2F, 0x21,
      0x94, 0xC4, 0xAF, 0x45, 0xA7, 0x62, 0x86, 0xDA, 0xB2, 0xB0, 0xE0,
      0xFE, 0x8D, 0xD3, 0x7A, 0xF7, 0x2E, 0xD0, 0x21, 0xAC, 0xA6},
     {0x58, 0x88, 0x9B, 0x43, 0x7C, 0xB7, 0xD1, 0x60, 0xEB, 0xE5, 0x3B,
      0x11, 0x05, 0xA6, 0xE2, 0xCB, 0x55, 0xBD, 0x41, 0x2D, 0x37, 0xCE,
      0x68, 0x37, 0x21, 0xCD, 0xF0, 0xFA, 0x73, 0x36, 0x60, 0xC1}},

    {params_ml_kem768,
     {0x85, 0x90, 0xBF, 0xC9, 0xA6, 0xFC, 0x25, 0xEE, 0x7E, 0x6D, 0xAB,
      0x48, 0x70, 0xDB, 0xF4, 0xB5, 0x1A, 0x1F, 0x14, 0x1B, 0x7C, 0x9E,
      0x96, 0x23, 0x0C, 0x04, 0x03, 0xE7, 0x99, 0xBC, 0x68, 0xE0, 0xBF,
      0x83, 0xE3, 0x04, 0x8B, 0x02, 0x1F, 0x22, 0xDB, 0x57, 0x07, 0x6A,
      0x88, 0x57, 0x29, 0xF9, 0x51, 0x19, 0xCE, 0x63, 0xFA, 0xF5, 0x1A,
      0x69, 0x95, 0x4B, 0xCC, 0xC5, 0x1E, 0x01, 0x46, 0x86},
     {0xC5, 0x7B, 0x98, 0x07, 0x58, 0x6D, 0xB3, 0xD9, 0x9C, 0x6A, 0xFF,
      0xAF, 0xB0, 0x4C, 0xD2, 0x55, 0x1A, 0x4B, 0x1D, 0xF1, 0x7F, 0xCC,
      0xB8, 0xD7, 0xD9, 0x4C, 0x10, 0x3E, 0xE6, 0x65, 0x6B, 0x14},
     {0x07, 0xA3, 0xC2, 0x9D, 0xC6, 0xBE, 0x44, 0xB8, 0x0E, 0x1C, 0x77,
      0xB0, 0x56, 0x8D, 0x48, 0x19, 0xE6, 0xBE, 0x1C, 0x89, 0x98, 0x31,
      0x9D, 0x24, 0x29, 0xF9, 0x13, 0x78, 0x00, 0x63, 0x6C, 0xE4}},

    {params_ml_kem768,
     {0xD5, 0xFD, 0x81, 0x50, 0x92, 0x62, 0x0D, 0xC4, 0x2A, 0x22, 0x39,
      0x09, 0xE3, 0x87, 0x36, 0x9A, 0x74, 0xAF, 0x7D, 0xCA, 0x28, 0x51,
      0x38, 0xCF, 0x21, 0x7B, 0xC2, 0x9F, 0x29, 0xC4, 0x2C, 0x41, 0xF4,
      0x28, 0x61, 0xEF, 0xF7, 0x69, 0x16, 0x14, 0xC3, 0xE8, 0x97, 0x5A,
      0xFB, 0x4E, 0x35, 0x3F, 0x8C, 0x8C, 0x39, 0xE6, 0xF4, 0x1B, 0xB6,
      0x37, 0xEC, 0x79, 0xBA, 0xA9, 0x76, 0xD1, 0xAD, 0xC1},
     {0xE8, 0xBE, 0xB5, 0xE4, 0x0D, 0xA1, 0x6C, 0xD0, 0xB6, 0x77, 0x1A,
      0x00, 0x6B, 0xD6, 0xCC, 0x2A, 0x5B, 0xA7, 0x7C, 0x27, 0x8E, 0x3E,
      0xDF, 0x52, 0x91, 0x22, 0x10, 0xF8, 0x0A, 0x5E, 0x17, 0x59},
     {0x52, 0x77, 0x12, 0x00, 0xD2, 0x28, 0x97, 0x43, 0xED, 0x59, 0xB9,
      0xFC, 0x37, 0xA7, 0xFB, 0xC4, 0xC2, 0x17, 0x7A, 0x4A, 0x96, 0xFB,
      0x9D, 0x6A, 0x02, 0x12, 0x22, 0xFE, 0xE5, 0xE3, 0x9E, 0xB7}},

    {params_ml_kem768,
     {0xD2, 0x1D, 0x5A, 0xFE, 0xD9, 0xAF, 0xAA, 0x3B, 0x49, 0xFB, 0x45,
      0x24, 0x5B, 0x2B, 0xCA, 0x15, 0x05, 0xE4, 0x00, 0x0C, 0xDC, 0x29,
      0x09, 0x4A, 0x36, 0x00, 0xF5, 0xCA, 0xA4, 0x9A, 0x7B, 0x3A, 0x4D,
      0xD0, 0xE8, 0x60, 0x91, 0x64, 0x9A, 0x0A, 0x08, 0xEA, 0x44, 0xDA,
      0xB8, 0x5D, 0xF5, 0x67, 0x97, 0xF8, 0xBF, 0x46, 0x22, 0x2C, 0x2D,
      0xBA, 0x7D, 0xEC, 0x63, 0x74, 0xB9, 0xB2, 0x26, 0x8E},
     {0x6C, 0x77, 0x0D, 0x1F, 0xA4, 0xC0, 0xF5, 0xDB, 0xB6, 0x60, 0x53,
      0x07, 0x72, 0xFC, 0xC2, 0x29, 0x7F, 0x59, 0xBC, 0x9D, 0xEE, 0x33,
      0x8C, 0xD1, 0x24, 0xF0, 0x92, 0x4C, 0xF7, 0xE3, 0x76, 0x2D},
     {0xBF, 0x2B, 0x7F, 0x98, 0xC1, 0x4F, 0x13, 0x63, 0x3F, 0x97, 0x8A,
      0x89, 0x7B, 0x39, 0x45, 0xC8, 0x55, 0xB4, 0x2F, 0x95, 0xBC, 0x6C,
      0x97, 0x43, 0x0A, 0x15, 0x4D, 0x51, 0x43, 0x37, 0x5C, 0x48}}};
